

#include "externs.h"

void
write_8250(uint32_t regaddr, uint8_t val)
{
	volatile unsigned char *reg;
	reg = (unsigned char*)regaddr;
	reg[0] = val;
}

uint8_t read_8250(uint32_t regaddr)
{
	volatile unsigned char *reg;	
	reg = (unsigned char*)regaddr;
	return reg[0]; 
}

void
set_port(unsigned port, unsigned mask, unsigned data) {
	unsigned char c;

	c = read_8250(port);
	write_8250(port, (c & ~mask) | (data & mask));
}

static void
clear_device(const uintptr_t *port) {
	unsigned char tmp;

	write_8250(port[REG_IE], 0);					// Disable all interrupts
	tmp = read_8250(port[REG_LS]);						// Clear Line Status Interrupt
	tmp = read_8250(port[REG_RX]);						// Clear RX Interrupt
	tmp = read_8250(port[REG_TX]);						// Clear TX Interrupt
	tmp = read_8250(port[REG_MS]);						// Clear Modem Interrupt
}

//
// Clean up the device then add it to the interrupt list and enable it.
//
void
ser_attach_intr(DEV_8250 *dev) {
	uintptr_t	*port = dev->port;
	struct dev_list	**owner;
	struct dev_list *curr;

	// Clean the device so we get a level change on the intr line to the bus.
	// Enable out2 (gate intr to bus)
	set_port(port[REG_MC], MCR_OUT2, MCR_OUT2);
	clear_device(port);

	// Add it to the interrupt list
	owner = &devices;
	for( ;; ) {
		curr = *owner;
		if(curr == NULL) {
			curr = _smalloc(sizeof(*curr));
			*owner = curr;
			curr->next = NULL;
			curr->device = NULL;
			break;
		}
		if(curr->device->intr == dev->intr) break;
		owner = &curr->next;
	}
	
	// Delay interrupts while we're fiddling around with the list
	InterruptMask(dev->intr, -1);
	dev->next = curr->device;
	curr->device = dev;
	InterruptUnmask(dev->intr, -1);

	// If first handler, attach the interrupt.
	if(curr->device->next == NULL) {		
		curr->iid = InterruptAttach(_NTO_INTR_CLASS_EXTERNAL | dev->intr, ser_intr, curr, 0, _NTO_INTR_FLAGS_END);	
	}

	// Enable ALL interrupt sources.
	write_8250(port[REG_IE], 0x0f);	
}

void
create_device(TTYINIT *dip, unsigned unit, unsigned int dev_id) {
	DEV_8250 			*dev;
	unsigned			i;
	uintptr_t port_regbase;
	unsigned char		msr;	

	// Get a device entry and the input/output buffers for it.
	dev = (void *) _smalloc(sizeof(*dev));
	memset(dev, 0, sizeof(*dev));
	
	dev->dev_id = dev_id;

	// Get buffers.
	dev->tty.ibuf.head = dev->tty.ibuf.tail = dev->tty.ibuf.buff = _smalloc(dev->tty.ibuf.size = dip->isize);
	dev->tty.obuf.head = dev->tty.obuf.tail = dev->tty.obuf.buff = _smalloc(dev->tty.obuf.size = dip->osize);
	dev->tty.cbuf.head = dev->tty.cbuf.tail = dev->tty.cbuf.buff = _smalloc(dev->tty.cbuf.size = dip->csize);
	dev->tty.highwater = dev->tty.ibuf.size - (dev->tty.ibuf.size < 128 ? dev->tty.ibuf.size/4 : 32);

	strcpy(dev->tty.name, dip->name);

	dev->tty.baud = dip->baud;
	if( dev_id > 0x258 ) // for 0x35x- PCie
		dev->tty.fifo = 0xFF; // -ravi - shoould have been 256, but tty.fifo is an unsigned char, so, just fill in the maximum for a char; this variable is not used anyway
	else
		dev->tty.fifo = dip->fifo;

	// If TX fifo is set, set the XMIT override flag
//ravi	if(dev->tty.fifo & 0xF0) {
		dev->fifo_override = FIFO_XMIT_OVERRIDE; //ravi - setting this flag is needed for TXing data because of the logic in tto(). 		 
//	}

	if( dev_id > 0x258 ) // for 0x35x- PCie
		port_regbase = mmap_device_memory(NULL, 0x400, PROT_READ|PROT_WRITE|PROT_NOCACHE, 0, dip->port);
	else
		port_regbase = mmap_device_memory(NULL, 0x200, PROT_READ|PROT_WRITE|PROT_NOCACHE, 0, dip->port);
		
	for(i = 0; i < sizeof(dev->port)/sizeof(dev->port[0]); ++i) {
		dev->port[i] = port_regbase;
		port_regbase += 1;		
	}

	dev->intr = dip->intr;
	if( dev_id > 0x258 ) // for 0x35x- PCie
		dev->clk = 125000000; // 125 MHz
	else if( dev_id > 0x158 ) // for 0x25x
		dev->clk = 24000000; // 24 MHz
	else
		dev->clk = dip->clk; // default is 14.7 MHz	
	dev->div = dip->div;

	dev->tty.flags = EDIT_INSERT | LOSES_TX_INTR;
	dev->tty.c_cflag = dip->c_cflag;
	dev->tty.c_iflag = dip->c_iflag;
	dev->tty.c_lflag = dip->c_lflag;
	dev->tty.c_oflag = dip->c_oflag;

	// Initialize termios cc codes to an ANSI terminal.
	ttc(TTC_INIT_CC, &dev->tty, 0);

	// Initialize the device's name.
	// Assume that the basename is set in device name.  This will attach
	// to the path assigned by the unit number/minor number combination
	unit = SET_NAME_NUMBER(unit) | NUMBER_DEV_FROM_USER;
	ttc(TTC_INIT_TTYNAME, &dev->tty, unit);

	// Initialize power management structures before attaching ISR
	ttc(TTC_INIT_POWER, &dev->tty, 0);

	// Only setup IRQ handler for non-pcmcia devices.
	// Pcmcia devices will have this done later when card is inserted.
	if(dip->port != 0 && dip->intr != _NTO_INTR_SPARE) {
		ser_stty(dev);
		ser_attach_intr(dev);
	}

	// get current MSR stat
	msr = read_8250(dev->port[REG_MS]);

	if(msr & MSR_DDCD)
		tti(&dev->tty, (msr & MSR_DCD) ? TTI_CARRIER : TTI_HANGUP);
					
	if((msr & MSR_DCTS)  &&  (dev->tty.c_cflag & OHFLOW))
		tti(&dev->tty, (msr & MSR_CTS) ? TTI_OHW_CONT : TTI_OHW_STOP);

	// Attach the resource manager
	ttc(TTC_INIT_ATTACH, &dev->tty, 0);

}
