

#include "externs.h"

int
tto(TTYDEV *ttydev, int action, int arg1) {
	TTYBUF 			*bup = &ttydev->obuf;
	DEV_8250		*dev = (DEV_8250 *)ttydev;
	const uintptr_t	*port = dev->port;
	int 			status = 0;
	unsigned char 	c;

	switch(action) {
	case TTO_STTY:
		ser_stty(dev);
		return(0);

	case TTO_CTRL:

		if(arg1 & _SERCTL_BRK_CHG)
			set_port(port[REG_LC], LCR_BREAK, arg1 &_SERCTL_BRK ? LCR_BREAK : 0);

		if(arg1 & _SERCTL_DTR_CHG)
			set_port(port[REG_MC], MCR_DTR, arg1 & _SERCTL_DTR ? MCR_DTR : 0);

		if(arg1 & _SERCTL_RTS_CHG)
			set_port(port[REG_MC], MCR_RTS, arg1 & _SERCTL_RTS ? MCR_RTS : 0);

		return(0);

	case TTO_LINESTATUS:
		return(((read_8250(port[REG_MS]) << 8) | read_8250(port[REG_MC])) & 0xf003);

	case TTO_DATA:
		break;

	default:
		return(0);
	}

   /*
    * If the OSW_PAGED_OVERRIDE flag is set then allow
    * transmit of character even if output is suspended via
    * the OSW_PAGED flag. This flag implies that the next
    * character in the obuf is a software flow control
    * charater (STOP/START).
    * Note: tx_inject sets it up so that the contol
    *       character is at the start (tail) of the buffer.
    *
   */
   if(dev->tty.flags & (OHW_PAGED|OSW_PAGED) && !(dev->tty.xflags & OSW_PAGED_OVERRIDE))
		return(0);

	dev_lock(&dev->tty);
	if(bup->cnt > 0) {
		if( (read_8250(port[REG_LS]) & LSR_TXRDY) || ( dev->tx_fifo > 0 && arg1 == FIFO_XMIT_OVERRIDE) ) {
			// Get the next character to print from the output buffer
			c = *bup->tail;
			if(c == '\n'  &&  ((dev->tty.c_oflag & (OPOST | ONLCR)) == (OPOST | ONLCR)) && 
					((dev->tty.flags & NL_INSERT) == 0)) {
				c = '\r';
				atomic_set(&dev->tty.flags, NL_INSERT);
			} else {
				atomic_clr(&dev->tty.flags, NL_INSERT);
				if(++bup->tail >= &bup->buff[bup->size])
					bup->tail = &bup->buff[0];
				--bup->cnt;
			}
	
			// Print the character
			dev->tty.un.s.tx_tmr = 3;  /* Timeout */
			write_8250(port[REG_TX], c);
         /* Clear the OSW_PAGED_OVERRIDE flag as we only want
          * one character to be transmitted in this case.
          */
         if (dev->tty.xflags & OSW_PAGED_OVERRIDE)
            atomic_clr(&dev->tty.xflags, OSW_PAGED_OVERRIDE);
		}
		dev_unlock(&dev->tty);
	} else {
		dev_unlock(&dev->tty);

		// Check for notify conditions
		if(dev->tty.notify[1].cnt < bup->size - bup->cnt) {
			dev->tty.notify[1].cnt = (~0u) >> 1;	// Disarm
			atomic_set(&dev->tty.flags, EVENT_NOTIFY_OUTPUT);
			status = 1;
		}

		// Is anyone waiting for the output buffer to drain?
		if(dev->tty.waiting_drain && bup->cnt == 0) {
			atomic_set(&dev->tty.flags, EVENT_DRAIN);
			status = 1;
		}
	}

	// If anyone is waiting to write, kick them when buffer drains to 1/4 full.
	if(dev->tty.waiting_write && bup->cnt < bup->size/4) {
		atomic_set(&dev->tty.flags, EVENT_WRITE);
		return(1);
	}

	return(status);
}

void
ser_stty(DEV_8250 *dev) {
	unsigned 		lcr = 0;
	const uintptr_t *port = dev->port;
	unsigned		value;

	// Set Baud rate
	value = (dev->tty.baud == 0) ? 0 : (dev->clk/(dev->tty.baud * dev->div));

	dev_lock(&dev->tty);
	if( dev->dev_id > 0x158 ) // if 0x25x or 0x35x
	{
		unsigned int quot_fraction;
		quot_fraction = (dev->clk / dev->tty.baud) - (dev->div * value);
		set_port(port[REG_LC], LCR_DLAB, LCR_DLAB);
		set_port(port[REG_DL0], 0xff, value & 0xff);
		set_port(port[REG_DL1], 0xff, value >> 8);		
		write_8250(port[XR_DIVISOR_LATCH_FRACTION], (quot_fraction & 0xf));
		set_port(port[REG_LC], LCR_DLAB, 0);
	}	
	else
	{
		set_port(port[REG_LC], LCR_DLAB, LCR_DLAB);
		set_port(port[REG_DL0], 0xff, value & 0xff);
		set_port(port[REG_DL1], 0xff, value >> 8);
		set_port(port[REG_LC], LCR_DLAB, 0);		
	}
	
	dev_unlock(&dev->tty);

	// Set data bits
	switch(dev->tty.c_cflag & CSIZE) {
	case CS8: ++lcr;
	case CS7: ++lcr;
	case CS6: ++lcr;
	}

	// Set stop bits
	if(dev->tty.c_cflag & CSTOPB)
		lcr |= LCR_STB2;

	// Set parity bits
	if(dev->tty.c_cflag & PARENB)
		lcr |= LCR_PEN;

	if((dev->tty.c_cflag & PARODD) == 0)
		lcr |= LCR_EPS;

	set_port(port[REG_LC], 0xFF, lcr);

	// turn on DTR, RTS
	set_port(port[REG_MC], MCR_DTR|MCR_RTS, MCR_DTR|MCR_RTS);

	//ravi
	if( dev->dev_id > 0x258 ) // if PCIe-0x35x
		dev->rx_fifo = dev->tx_fifo = 256;
	else
		dev->rx_fifo = dev->tx_fifo = 64;
	write_8250(port[REG_FC], FCR_FIFO_ENABLE | FCR_RX_FIFO_RESET | FCR_TX_FIFO_RESET); // enable
	//setup optimal trigger levels
	unsigned char oldFCTR;
	oldFCTR = read_8250(port[XR_EXTENDED_FCTR]);
	//default Hysteresis of 16
	oldFCTR &= 0xf0; oldFCTR |= 0xc5;
	write_8250(port[XR_EXTENDED_FCTR], oldFCTR);
	if( dev->dev_id > 0x258 ) // if 0x25x or 0x35x
	{
		write_8250(port[XR_TXFIFOTRIG_REGISTER], 128); // 50% of 256 (FIFO length)
		write_8250(port[XR_RXFIFOTRIG_REGISTER], 128); // 50% of 256 (FIFO length)
	}
	else
	{
		write_8250(port[XR_TXFIFOTRIG_REGISTER], 32); // 50% of 64 (FIFO length)
		write_8250(port[XR_RXFIFOTRIG_REGISTER], 32); // 50% of 64 (FIFO length)
	}
}

int drain_check(TTYDEV *ttydev, uintptr_t *count) {
	TTYBUF 			*bup = &ttydev->obuf;
	DEV_8250		*dev = (DEV_8250 *)ttydev;
	const uintptr_t	*port = dev->port;

	// if the device has DRAINED, return 1
	if (dev->tty.waiting_drain && (bup->cnt == 0) &&
		(read_8250(port[REG_LS]) & LSR_TSRE)) return 1;

	// if the device has not DRAINED, set a timer based on 50ms counts
	// wait for the time it takes for one character to be transmitted
	// out the shift register.  We do this dynamically since the
	// baud rate can change.		
	*count = ((IO_CHAR_DEFAULT_BITSIZE * 20) / ttydev->baud) + 1;
	
	return 0;
}

